#! /usr/bin/env sh
# This script is part of the eix project and distributed under the
# terms of the GNU General Public License v2.
#
# Author and Copyright (c):
#   Martin V\"ath <vaeth@mathematik.uni-wuerzburg.de>
#
# This file must be "source"d by POSIX scripts.
#
# It contains helper functions for eix-remote, eix-layman,
# and perhaps similar local scripts. (eix 0.22.5).

# This must happen in a function to enforce an empty argument list.
ReadFunctions() {
	ReadGettext
	[ "${local_eprefix_portage_exec-unset}" = 'unset' ] &&
		ReadVar local_eprefix_portage_exec EPREFIX_PORTAGE_EXEC
	for eixf_i in "${local_eprefix_portage_exec}/etc/init.d/functions.sh" \
		"${local_eprefix_portage_exec}/sbin/functions.sh"
	do	test -r "${eixf_i}" || continue
		. "${eixf_i}"
		return
	done
	printf '%s: %s' "${0##*/}" "`gettext 'cannot read functions.sh'`" >&2
	exit 2
}

: ${read_var_prg:=eix-update}
ReadVar() {
	eixf_read=`PRINT_APPEND='X' "${3:-${read_var_prg}}" --print "${2}"`
	eval "${1}=\"\${eixf_read%X}\" && [ \"\${1}\" != \"\${eixf_read}\" ]"
}

die() {
	eerror "${*}"
	exitcode=2
	exit "${exitcode}"
}

PerLine() {
	while IFS='
' read eixf_line
	do	"${@}" "${eixf_line}"
	done
}

GetPortdir() {
	[ -n "${local_portdir}" ] || ReadVar local_portdir PORTDIR
}

verbose=:

RunCommand() {
	if ${verbose}
	then	einfo "${1}"
		shift
		"${@}"
		return ${?}
	fi
	ebegin "${1}"
	shift
	"${@}" >/dev/null
	eixf_ret=${?}
	eixf_cmd="${*}"
	eend ${eixf_ret} "`eval_gettext 'Problems running ${eixf_cmd}'`"
	return ${eixf_ret}
}

if ( eval '[ "$(( 0 + 1 ))" = 1 ]' ) >/dev/null 2>&1
then	eval '
Expr() {
	echo "$(( ${1} ${2} ${3} ))"
}'
else
Expr() {
	expr "${@}"
}
fi

# Usage: Replace [-g] VAR Search Replace
# Option -g means: Replace all occurrences
# Exit status is 0 if at least one replacement took place, and
# $replace_count contains at string of length(number of replacements) x'en.
Replace() {
	eixf_repla=''
	eixf_replc='break'
	case "${1}" in
	-*)	eixf_replc=''; shift;;
	esac
	replace_count=''
	eval "while {
		eixf_replb=\${${1}%%\"\${2}\"*}
		[ \"\${eixf_replb}\" != \"\${${1}}\" ]
	}
	do	eixf_repla=\"\${eixf_repla}\${eixf_replb}\${3}\"
		${1}=\${${1}#*\"\${2}\"}
		replace_count=\"\${replace_count}x\"
		${eixf_replc}
	done
	${1}=\"\${eixf_repla}\${${1}}\""
	[ -n "${replace_count}" ]
}

# Usage: Push [-c] VAR Args
# Append all Args to VAR, quoted such that
#    eval "${VAR}" is 'safe' and groups Args correctly.
# With option -c, VAR is cleaned in advance.
Push() {
	case "${1}" in
	-*)	shift; eval "${1}=''";;
	esac
	eixf_pusha="${1}"
	shift
	eval "for eixf_pushb
	do	Replace -g eixf_pushb \"'\" \"'\\\\''\"
		if [ -z \"\${${eixf_pusha}}\" ]
		then	${eixf_pusha}=\"'\${eixf_pushb}'\"
		else	${eixf_pusha}=\"\${${eixf_pusha}} '\${eixf_pushb}'\"
		fi
	done"
}

AddUpdateArgs() {
	Push update_args "${@}"
}

AddUpdateQuoting() {
	eixf_aq="${1}"
	shift
	Replace -g eixf_aq '\' '\\'
	Replace -g eixf_aq '?' '\?'
	Replace -g eixf_aq '*' '\*'
	Replace -g eixf_aq '[' '\['
	AddUpdateArgs "${eixf_aq}" "${@}"
}

AddMethod() {
	AddUpdateArgs '-m'
	AddUpdateQuoting "${1}" "${2}"
}

AddRepoName() {
	AddUpdateArgs '-r' "${1}" "${2}"
}

AddOverlays() {
	for eixf_ao
	do	AddUpdateArgs '-a' "${eixf_ao}"
	done
}

AddExcludes() {
	for eixf_ax
	do	AddUpdateArgs '-x'
		AddUpdateQuoting "${eixf_ax}"
	done
}

ClearUpdateArgs() {
	update_args=''
}

: ${die_on_update_failure=:}
CallUpdate() {
	eval "set -- ${update_args}"
	RunCommand "`gettext 'Calling eix-update'`" eix-update "${@}" && return
	${die_on_update_failure} && die "`gettext 'eix-update failed'`"
}

AddLocalMethods() {
	GetPortdir
	AddMethod "${local_portdir}" "eix"
	for eixf_alm in `portageq portdir_overlay` `eix-update --print ADD_OVERLAY`
	do	eixf_al="${eixf_alm}"
		Replace -g eixf_al '\' '\\'
		Replace -g eixf_al ':' '\:'
		AddMethod "${eixf_alm}" "eix::${eixf_al}"
	done
}

# Remove optional trailing newline from all variables (arguments).
# This is necessary as some shells won't take ${Var%"[newline]"}
Chomp() {
	for eixf_chompa
	do	eval "eixf_chompb=\"\${${eixf_chompa}}\""
		case "${eixf_chompb}" in
			*'
')
			eixf_chompb="${eixf_chompb%?}"
			eval "${eixf_chompa}=\"\${eixf_chompb}\"";;
		esac
	done
}

# Consider variables (arguments) as commands and expand them to full path.
# Dies if a command cannot be found.

Pathify() {
	for eixf_pathc
	do	eval "eixf_patha=\"\${${eixf_pathc}}\""
		eixf_pathb=`command -v "${eixf_patha}"; printf A` && \
			eixf_pathb="${eixf_pathb%A}" && Chomp eixf_pathb && \
			[ -n "${eixf_pathb}" ] && test -x "${eixf_pathb}" || \
			die "`eval_gettext \
				'${eixf_patha} cannot be found in path'`"
		eval "${eixf_pathc}=\${eixf_pathb}"
	done
}

# Consider variables (arguments) as paths and normalize them.
# If normalize_resolve is : (default), use readlink/cd -P
# to resolve to "true" pathnames.
# Return with 1 if at least one variable is empty.
# Set relative to a list of all variables which are relative paths
# (the first token in relative will be a space if the list is not empty).

normalize_resolve=:
NormalizeNames() {
	${normalize_resolve} && if [ -z "${have_readlink}" ]
	then	command -v readlink >/dev/null 2>&1 \
			&& have_readlink=: || have_readlink=false
	fi
	eixf_normd=0
	relative=''
	for eixf_normc
	do	eval "eixf_norma=\"\${${eixf_normc}}\""
		if [ -z "${eixf_norma}" ]
		then	eixf_normd=1
			continue
		fi
		case "${eixf_norma}" in
			/*) :;;
			*) relative="${relative} ${eixf_normc}";;
		esac
		${normalize_resolve} && if ${have_readlink}
		then	# Append A to avoid loss of trailing space in `...`
			eixf_normb=`readlink -f -- "${eixf_norma}"; printf A` \
				&& eixf_normb="${eixf_normb%A}" \
				&& Chomp eixf_normb
		else	eixf_normb=`cd -P -- "${eixf_norma}" >/dev/null 2>&1 \
				&& printf '%sA' "${PWD}"` \
				&& eixf_normb="${eixf_normb%A}"
		fi && [ -n "${eixf_normb}" ] && eixf_norma="${eixf_normb}"
		eval "${eixf_normc}=\"\${eixf_norma}\""
		while Replace -g "${eixf_normc}" '//' '/'
		do :
		done
		eval "[ \"\${${eixf_normc}}\" = '/' ] || \
			${eixf_normc}=\"\${${eixf_normc}%/}\""
	done
	return ${eixf_normd}
}

ReadGettext() {
	[ -n "${read_gettext_status}" ] && return ${read_gettext_status}
	case "${USE_NLS-no}" in
	''|n*|N*|f*|F*|0)
		read_gettext_status=1;;
	*)	ReadGettextSub >/dev/null 2>&1
		read_gettext_status=${?};;
	esac
	if [ ${read_gettext_status} -eq 0 ]
	then	TEXTDOMAIN='eix'
		TEXTDOMAINDIR='/usr/share/locale'
		export TEXTDOMAIN TEXTDOMAINDIR
		return
	fi
	gettext() {
		printf '%s' "${*}"
	}
	eval_gettext() {
		eixf_gettext="${*}"
		Replace -g eixf_gettext '\' '\\'
		Replace -g eixf_gettext '$(' '\$('
		Replace -g eixf_gettext '`' '\`'
		Replace -g eixf_gettext '"' '\"'
		eval "printf '%s' \"${eixf_gettext}\""
	}
	ngettext() {
		gettext "${@}"
	}
	eval_ngettext() {
		eval_gettext "${@}"
	}
	return 1
}

ReadGettextSub() {
	. gettext.sh && return
	eixf_i="`command -v gettext.sh 2>/dev/null`" && \
		[ -n "${eixf_i}" ] && . "${eixf_i}" && return
	. /usr/bin/gettext.sh
}

AssignTempEmulate() {
	if [ -z "${have_random}" ]
	then	eixf_assb="${RANDOM}"
		if [ "${eixf_assb}" = "${RANDOM}" ] && \
			[ "${eixf_assb}" = "${RANDOM}" ]
		then	have_random=false
			eixf_assb=`od -d -N2 /dev/random 2>/dev/null` \
				|| eixf_assb=''
			eixf_assb=`echo ${eixf_assb}`
			eixf_assb="${eixf_assb#* }"
			eixf_assb="${eixf_assb%% *}"
			if [ -z "${eixf_assb}" ]
			then	eixf_assb=1
			else	eixf_assb=`Expr "${eixf_assb} % 32768"`
				[ "${eixf_assb}" -eq 0 ] && eixf_assb=1
			fi
		else	have_random=:
		fi
		eixf_assc=''
	fi
	eixf_assa=0
	while [ ${eixf_assa} -le 9999 ]
	do	if [ -n "${eixf_assc}" ]
		then	if ${have_random}
			then	eixf_assb="${RANDOM}"
			else	eixf_assb=`Expr "${eixf_assb}" '*' '13821'`
				eixf_assb=`Expr "${eixf_assb}" '%' '32768'`
			fi
		fi
		eixf_assc="/tmp/${0##*/}.${$}${eixf_assa}${eixf_assb}"
		if [ "${2}" = '-d' ]
		then	mkdir "${eixf_assc}"
		else	(
			set -o noclobber
			: >"${eixf_assc}"
		)
		fi && eval "${1}=\"\${eixf_assc}\"" && return
		eixf_assa="`Expr ${eixf_assa} + 1`"
	done
	die "`gettext 'found no free tempname after 10000 attempts'`"
}

AssignTemp() {
	eval "${1}=''"
	if [ -z "${have_mktemp}" ]
	then	command -v mktemp >/dev/null 2>&1 \
			&& have_mktemp=: || have_mktemp=false
	fi
	if ${have_mktemp}
	then	eval "${1}=\`mktemp ${2} \"/tmp/\${0##*/}.XXXXXXXX\"\`"
	else	AssignTempEmulate "${@}"
	fi && return
	[ "${2}" = '-d' ] && \
		die "`gettext 'cannot create temporary directory'`"
	die "`gettext 'cannot create temporary file'`"
}
